!    MCL: MiMiC Communication Library
!    Copyright (C) 2015-2025  The MiMiC Authors (see CONTRIBUTORS file for details).
!
!    This file is part of MCL.
!
!    MCL is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Lesser General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    MCL is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU Lesser General Public License for more details.
!
!    You should have received a copy of the GNU Lesser General Public License
!    along with this program.  If not, see <http://www.gnu.org/licenses/>.

!> Contains Fortran API functions.
module mcl

    use c_bindings

    implicit none

#include "requests.inc"

    public :: mcl_initialize
    public :: mcl_finalize
    public :: mcl_abort
    public :: mcl_get_api_version
    public :: mcl_get_num_programs
    public :: mcl_get_program_id
    public :: mcl_get_request_name
    public :: mcl_is_initialized
    public :: mcl_send
    public :: mcl_receive

    private :: send_char, &
               send_int32, send_int32_array1d, send_int32_array2d, send_int32_array3d, &
               send_int64, send_int64_array1d, send_int64_array2d, send_int64_array3d, &
               send_float32, send_float32_array1d, send_float32_array2d, send_float32_array3d, &
               send_float64, send_float64_array1d, send_float64_array2d, send_float64_array3d

    private :: recv_char, &
               recv_int32, recv_int32_array1d, recv_int32_array2d, recv_int32_array3d, &
               recv_int64, recv_int64_array1d, recv_int64_array2d, recv_int64_array3d, &
               recv_float32, recv_float32_array1d, recv_float32_array2d, recv_float32_array3d, &
               recv_float64, recv_float64_array1d, recv_float64_array2d, recv_float64_array3d

    !> Initializes communicator. For the MPI-based modes of communication, it should
    !> be called immediately after mpi_init.
    interface mcl_initialize
#ifdef USE_MPIF08
        module procedure mcl_init_f08
#endif
        module procedure mcl_init_f90
    end interface mcl_initialize

    !> Sends data to another program.
    interface mcl_send
        module procedure send_char, &
                         send_int32, send_int32_array1d, send_int32_array2d, send_int32_array3d, &
                         send_int64, send_int64_array1d, send_int64_array2d, send_int64_array3d, &
                         send_float32, send_float32_array1d, send_float32_array2d, send_float32_array3d, &
                         send_float64, send_float64_array1d, send_float64_array2d, send_float64_array3d
    end interface mcl_send

    !> Receives data from another program.
    interface mcl_receive
        module procedure recv_char, &
                         recv_int32, recv_int32_array1d, recv_int32_array2d, recv_int32_array3d, &
                         recv_int64, recv_int64_array1d, recv_int64_array2d, recv_int64_array3d, &
                         recv_float32, recv_float32_array1d, recv_float32_array2d, recv_float32_array3d, &
                         recv_float64, recv_float64_array1d, recv_float64_array2d, recv_float64_array3d
    end interface mcl_receive

contains

#ifdef USE_MPIF08
!> Initializes mpif08-type communicator (must be called after mpi_init).
subroutine mcl_init_f08(communicator, err)
    use iso_c_binding, only: c_int, c_loc
    use mpi_f08
    !> MPI communicator that is used by the host code
    type(mpi_comm), intent(inout), target :: communicator
    !> Error status.
    integer, optional, intent(out) :: err

    integer(kind=c_int) :: error

    error = c_mcl_initialize(c_loc(communicator))
    if (present(err)) err = error
end subroutine mcl_init_f08
#endif


!> Initializes mpif90-type communicator (must be called after mpi_init).
subroutine mcl_init_f90(communicator, err)
    use iso_c_binding, only: c_int, c_loc
    !> MPI communicator that is used by the host code.
    integer, intent(inout), target :: communicator
    !> Error status.
    integer, optional, intent(out) :: err

    integer(kind=c_int) :: error

    error = c_mcl_initialize(c_loc(communicator))
    if (present(err)) err = error
end subroutine mcl_init_f90


!> Finalizes the communicator.
subroutine mcl_finalize(err)
    use iso_c_binding, only: c_int
    !> Error status.
    integer, optional, intent(out) :: err

    integer(kind=c_int) :: error

    error = c_mcl_finalize()
    if (present(err)) err = error
end subroutine mcl_finalize


!> Terminates all processes.
subroutine mcl_abort(error_code, err)
    use iso_c_binding, only: c_int
    !> Error code to return.
    integer, intent(in) :: error_code
    !> Error status.
    integer, optional, intent(out) :: err

    integer(kind=c_int) :: error, c_error_code

    c_error_code = error_code
    error = c_mcl_abort(error_code)
    if (present(err)) err = error
end subroutine mcl_abort


!> Returns a version number of the library.
subroutine mcl_get_api_version(version)
    use iso_c_binding, only: c_loc, c_ptr
    use, intrinsic :: iso_fortran_env, only: int32
    !> Array to store the version number.
    integer, dimension(3), target :: version

    type(c_ptr) :: c_version

    c_version = c_loc(version)
    call c_mcl_get_api_version(c_version)
end subroutine mcl_get_api_version


!> Returns the number of programs.
subroutine mcl_get_num_programs(num)
    !> Number of programs.
    integer, intent(out) :: num

    num = c_mcl_get_num_programs()
end subroutine mcl_get_num_programs


!> Returns the invoking program ID.
subroutine mcl_get_program_id(id)
    !> Program ID.
    integer, intent(out) :: id

    id = c_mcl_get_program_id()
end subroutine mcl_get_program_id


!> Returns the initialization status of MCL.
subroutine mcl_is_initialized(is_initialized)
    use iso_c_binding, only: c_bool
    !> MCL status.
    logical, intent(out) :: is_initialized
    is_initialized = c_mcl_is_initialized()
end subroutine mcl_is_initialized


!> Returns the request name associated with the given request value.
function mcl_get_request_name(request) result(request_name)
    use iso_c_binding, only: c_char, c_int, c_ptr, c_null_char, c_f_pointer
    !> Request value.
    integer(kind=c_int), intent(in) :: request
    !> String with the request name.
    character(:), allocatable :: request_name

    type(c_ptr) :: c_string
    integer(kind=c_int) :: c_length
    character(len=:, kind=c_char), pointer :: c_chars

    c_string = c_mcl_get_request_name(request)
    c_length = int(c_strlen(c_string), kind=c_int)
    block
        character(len=c_length, kind=c_char), pointer :: chars
        call c_f_pointer(c_string, chars)
        c_chars => chars
    end block
    request_name = c_chars
end function mcl_get_request_name    


!> Sends character string.
subroutine send_char(buffer, length, tag, destination, err)
    use iso_c_binding, only: c_ptr, c_int, c_loc
    !> Pointer to the buffer to send data from.
    character(len=*), target :: buffer
    !> Type of data to be sent.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program that receives data.
    integer :: destination
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_destination
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_destination = destination

    error = c_mcl_send(c_buffer, c_length, MCL_CHAR, c_tag, c_destination)
    if (present(err)) err = error
end subroutine send_char


!> Sends 32-bit integer.
subroutine send_int32(buffer, length, tag, destination, err)
    use iso_c_binding, only: c_ptr, c_int, c_loc
    use, intrinsic :: iso_fortran_env, only: int32
    !> Pointer to the buffer to send data from.
    integer(kind=int32), target :: buffer
    !> Type of data to be sent.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program that receives data.
    integer :: destination
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_destination
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_destination = destination

    error = c_mcl_send(c_buffer, c_length, MCL_INT, c_tag, c_destination)
    if (present(err)) err = error
end subroutine send_int32

!> Sends 1-dimensional 32-bit integer array.
subroutine send_int32_array1d(buffer, length, tag, destination, err)
    use iso_c_binding, only: c_ptr, c_int, c_loc
    use, intrinsic :: iso_fortran_env, only: int32
    !> Pointer to the buffer to send data from.
    integer(kind=int32), dimension(:), target :: buffer
    !> Type of data to be sent.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program that receives data.
    integer :: destination
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_destination
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_destination = destination

    error = c_mcl_send(c_buffer, c_length, MCL_INT, c_tag, c_destination)
    if (present(err)) err = error
end subroutine send_int32_array1d

!> Sends 2-dimensional 32-bit integer array.
subroutine send_int32_array2d(buffer, length, tag, destination, err)
    use iso_c_binding, only: c_ptr, c_int, c_loc
    use, intrinsic :: iso_fortran_env, only: int32
    !> Pointer to the buffer to send data from.
    integer(kind=int32), dimension(:,:), target :: buffer
    !> Type of data to be sent.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program that receives data.
    integer :: destination
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_destination
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_destination = destination

    error = c_mcl_send(c_buffer, c_length, MCL_INT, c_tag, c_destination)
    if (present(err)) err = error
end subroutine send_int32_array2d

!> Sends 3-dimensional 32-bit integer array.
subroutine send_int32_array3d(buffer, length, tag, destination, err)
    use iso_c_binding, only: c_ptr, c_int, c_loc
    use, intrinsic :: iso_fortran_env, only: int32
    !> Pointer to the buffer to send data from.
    integer(kind=int32), dimension(:,:,:), target :: buffer
    !> Type of data to be sent.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program that receives data.
    integer :: destination
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_destination
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_destination = destination

    error = c_mcl_send(c_buffer, c_length, MCL_INT, c_tag, c_destination)
    if (present(err)) err = error
end subroutine send_int32_array3d


!> Sends 64-bit integer.
subroutine send_int64(buffer, length, tag, destination, err)
    use iso_c_binding, only: c_ptr, c_int, c_loc
    use, intrinsic :: iso_fortran_env, only: int64
    !> Pointer to the buffer to send data from.
    integer(kind=int64), target :: buffer
    !> Type of data to be sent.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program that receives data.
    integer :: destination
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_destination
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_destination = destination

    error = c_mcl_send(c_buffer, c_length, MCL_LONG_INT, c_tag, c_destination)
    if (present(err)) err = error
end subroutine send_int64

!> Sends 1-dimensional 64-bit integer array.
subroutine send_int64_array1d(buffer, length, tag, destination, err)
    use iso_c_binding, only: c_ptr, c_int, c_loc
    use, intrinsic :: iso_fortran_env, only: int64
    !> Pointer to the buffer to send data from.
    integer(kind=int64), dimension(:), target :: buffer
    !> Type of data to be sent.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program that receives data.
    integer :: destination
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_destination
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_destination = destination

    error = c_mcl_send(c_buffer, c_length, MCL_LONG_INT, c_tag, c_destination)
    if (present(err)) err = error
end subroutine send_int64_array1d

!> Sends 2-dimensional 64-bit integer array.
subroutine send_int64_array2d(buffer, length, tag, destination, err)
    use iso_c_binding, only: c_ptr, c_int, c_loc
    use, intrinsic :: iso_fortran_env, only: int64
    !> Pointer to the buffer to send data from.
    integer(kind=int64), dimension(:,:), target :: buffer
    !> Type of data to be sent.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program that receives data.
    integer :: destination
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_destination
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_destination = destination

    error = c_mcl_send(c_buffer, c_length, MCL_LONG_INT, c_tag, c_destination)
    if (present(err)) err = error
end subroutine send_int64_array2d

!> Sends 3-dimensional 64-bit integer array.
subroutine send_int64_array3d(buffer, length, tag, destination, err)
    use iso_c_binding, only: c_ptr, c_int, c_loc
    use, intrinsic :: iso_fortran_env, only: int64
    !> Pointer to the buffer to send data from.
    integer(kind=int64), dimension(:,:,:), target :: buffer
    !> Type of data to be sent.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program that receives data.
    integer :: destination
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_destination
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_destination = destination

    error = c_mcl_send(c_buffer, c_length, MCL_LONG_INT, c_tag, c_destination)
    if (present(err)) err = error
end subroutine send_int64_array3d


!> Sends single-precision floating point.
subroutine send_float32(buffer, length, tag, destination, err)
    use, intrinsic :: iso_fortran_env, only: real32
    use iso_c_binding, only: c_ptr, c_int, c_loc
    !> Pointer to the buffer to send data from.
    real(kind=real32), target :: buffer
    !> Type of data to be sent.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program that receives data.
    integer :: destination
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_destination
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_destination = destination

    error = c_mcl_send(c_buffer, c_length, MCL_FLOAT, c_tag, c_destination)
    if (present(err)) err = error
end subroutine send_float32

!> Sends 1-dimensional single-precision floating point array.
subroutine send_float32_array1d(buffer, length, tag, destination, err)
    use, intrinsic :: iso_fortran_env, only: real32
    use iso_c_binding, only: c_ptr, c_int, c_loc
    !> Pointer to the buffer to send data from.
    real(kind=real32), dimension(:), target :: buffer
    !> Type of data to be sent.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program that receives data.
    integer :: destination
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_destination
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_destination = destination

    error = c_mcl_send(c_buffer, c_length, MCL_FLOAT, c_tag, c_destination)
    if (present(err)) err = error
end subroutine send_float32_array1d

!> Sends 2-dimensional single-precision floating point array.
subroutine send_float32_array2d(buffer, length, tag, destination, err)
    use, intrinsic :: iso_fortran_env, only: real32
    use iso_c_binding, only: c_ptr, c_int, c_loc
    !> Pointer to the buffer to send data from.
    real(kind=real32), dimension(:,:), target :: buffer
    !> Type of data to be sent.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program that receives data.
    integer :: destination
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_destination
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_destination = destination

    error = c_mcl_send(c_buffer, c_length, MCL_FLOAT, c_tag, c_destination)
    if (present(err)) err = error
end subroutine send_float32_array2d

!> Sends 3-dimensional single-precision floating point array.
subroutine send_float32_array3d(buffer, length, tag, destination, err)
    use, intrinsic :: iso_fortran_env, only: real32
    use iso_c_binding, only: c_ptr, c_int, c_loc
    !> Pointer to the buffer to send data from.
    real(kind=real32), dimension(:,:,:), target :: buffer
    !> Type of data to be sent.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program that receives data.
    integer :: destination
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_destination
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_destination = destination

    error = c_mcl_send(c_buffer, c_length, MCL_FLOAT, c_tag, c_destination)
    if (present(err)) err = error
end subroutine send_float32_array3d


!> Sends double-precision floating point.
subroutine send_float64(buffer, length, tag, destination, err)
    use, intrinsic :: iso_fortran_env, only: real64
    use iso_c_binding, only: c_ptr, c_int, c_loc
    !> Pointer to the buffer to send data from.
    real(kind=real64), target :: buffer
    !> Type of data to be sent.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program that receives data.
    integer :: destination
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_destination
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_destination = destination

    error = c_mcl_send(c_buffer, c_length, MCL_DOUBLE, c_tag, c_destination)
    if (present(err)) err = error
end subroutine send_float64

!> Sends 1-dimensional double-precision floating point array.
subroutine send_float64_array1d(buffer, length, tag, destination, err)
    use, intrinsic :: iso_fortran_env, only: real64
    use iso_c_binding, only: c_ptr, c_int, c_loc
    !> Pointer to the buffer to send data from.
    real(kind=real64), dimension(:), target :: buffer
    !> Type of data to be sent.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program that receives data.
    integer :: destination
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_destination
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_destination = destination

    error = c_mcl_send(c_buffer, c_length, MCL_DOUBLE, c_tag, c_destination)
    if (present(err)) err = error
end subroutine send_float64_array1d

!> Sends 2-dimensional double-precision floating point array.
subroutine send_float64_array2d(buffer, length, tag, destination, err)
    use, intrinsic :: iso_fortran_env, only: real64
    use iso_c_binding, only: c_ptr, c_int, c_loc
    !> Pointer to the buffer to send data from.
    real(kind=real64), dimension(:,:), target :: buffer
    !> Type of data to be sent.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program that receives data.
    integer :: destination
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_destination
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_destination = destination

    error = c_mcl_send(c_buffer, c_length, MCL_DOUBLE, c_tag, c_destination)
    if (present(err)) err = error
end subroutine send_float64_array2d

!> Sends 3-dimensional double-precision floating point array.
subroutine send_float64_array3d(buffer, length, tag, destination, err)
    use, intrinsic :: iso_fortran_env, only: real64
    use iso_c_binding, only: c_ptr, c_int, c_loc
    !> Pointer to the buffer to send data from.
    real(kind=real64), dimension(:,:,:), target :: buffer
    !> Type of data to be sent.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program that receives data.
    integer :: destination
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_destination
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_destination = destination

    error = c_mcl_send(c_buffer, c_length, MCL_DOUBLE, c_tag, c_destination)
    if (present(err)) err = error
end subroutine send_float64_array3d


!> Receives character string.
subroutine recv_char(buffer, length, tag, source, err)
    use iso_c_binding, only: c_ptr, c_int, c_loc
    !> Pointer to the buffer to receive data.
    character(len=*), target :: buffer
    !> Size of data buffer.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program to receive data from.
    integer :: source
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_source
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_source = source

    error = c_mcl_receive(c_buffer, c_length, MCL_CHAR, c_tag, c_source)
    if (present(err)) err = error
end subroutine recv_char


!> Receives 32-bit integer.
subroutine recv_int32(buffer, length, tag, source, err)
    use iso_c_binding, only: c_ptr, c_int, c_loc
    use, intrinsic :: iso_fortran_env, only: int32
    !> Pointer to the buffer to receive data.
    integer(kind=int32), target :: buffer
    !> Size of data buffer.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program to receive data from.
    integer :: source
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_source
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_source = source

    error = c_mcl_receive(c_buffer, c_length, MCL_INT, c_tag, c_source)
    if (present(err)) err = error
end subroutine recv_int32

!> Receives 1-dimensional 32-bit integer array.
subroutine recv_int32_array1d(buffer, length, tag, source, err)
    use iso_c_binding, only: c_ptr, c_int, c_loc
    use, intrinsic :: iso_fortran_env, only: int32
    !> Pointer to the buffer to receive data.
    integer(kind=int32), dimension(:), target :: buffer
    !> Size of data buffer.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program to receive data from.
    integer :: source
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_source
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_source = source

    error = c_mcl_receive(c_buffer, c_length, MCL_INT, c_tag, c_source)
    if (present(err)) err = error
end subroutine recv_int32_array1d

!> Receives 2-dimensional 32-bit integer array.
subroutine recv_int32_array2d(buffer, length, tag, source, err)
    use iso_c_binding, only: c_ptr, c_int, c_loc
    use, intrinsic :: iso_fortran_env, only: int32
    !> Pointer to the buffer to receive data.
    integer(kind=int32), dimension(:,:), target :: buffer
    !> Size of data buffer.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program to receive data from.
    integer :: source
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_source
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_source = source

    error = c_mcl_receive(c_buffer, c_length, MCL_INT, c_tag, c_source)
    if (present(err)) err = error
end subroutine recv_int32_array2d

!> Receives 3-dimensional 32-bit integer array.
subroutine recv_int32_array3d(buffer, length, tag, source, err)
    use iso_c_binding, only: c_ptr, c_int, c_loc
    use, intrinsic :: iso_fortran_env, only: int32
    !> Pointer to the buffer to receive data.
    integer(kind=int32), dimension(:,:,:), target :: buffer
    !> Size of data buffer.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program to receive data from.
    integer :: source
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_source
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_source = source

    error = c_mcl_receive(c_buffer, c_length, MCL_INT, c_tag, c_source)
    if (present(err)) err = error
end subroutine recv_int32_array3d


!> Receives 64-bit integer.
subroutine recv_int64(buffer, length, tag, source, err)
    use iso_c_binding, only: c_ptr, c_int, c_loc
    use, intrinsic :: iso_fortran_env, only: int64
    !> Pointer to the buffer to receive data.
    integer(kind=int64), target :: buffer
    !> Size of data buffer.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program to receive data from.
    integer :: source
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_source
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_source = source

    error = c_mcl_receive(c_buffer, c_length, MCL_LONG_INT, c_tag, c_source)
    if (present(err)) err = error
end subroutine recv_int64

!> Receives 1-dimensional 64-bit integer array.
subroutine recv_int64_array1d(buffer, length, tag, source, err)
    use iso_c_binding, only: c_ptr, c_int, c_loc
    use, intrinsic :: iso_fortran_env, only: int64
    !> Pointer to the buffer to receive data.
    integer(kind=int64), dimension(:), target :: buffer
    !> Size of data buffer.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program to receive data from.
    integer :: source
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_source
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_source = source

    error = c_mcl_receive(c_buffer, c_length, MCL_LONG_INT, c_tag, c_source)
    if (present(err)) err = error
end subroutine recv_int64_array1d

!> Receives 2-dimensional 64-bit integer array.
subroutine recv_int64_array2d(buffer, length, tag, source, err)
    use iso_c_binding, only: c_ptr, c_int, c_loc
    use, intrinsic :: iso_fortran_env, only: int64
    !> Pointer to the buffer to receive data.
    integer(kind=int64), dimension(:,:), target :: buffer
    !> Size of data buffer.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program to receive data from.
    integer :: source
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_source
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_source = source

    error = c_mcl_receive(c_buffer, c_length, MCL_LONG_INT, c_tag, c_source)
    if (present(err)) err = error
end subroutine recv_int64_array2d

!> Receives 3-dimensional 64-bit integer array.
subroutine recv_int64_array3d(buffer, length, tag, source, err)
    use iso_c_binding, only: c_ptr, c_int, c_loc
    use, intrinsic :: iso_fortran_env, only: int64
    !> Pointer to the buffer to receive data.
    integer(kind=int64), dimension(:,:,:), target :: buffer
    !> Size of data buffer.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program to receive data from.
    integer :: source
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_source
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_source = source

    error = c_mcl_receive(c_buffer, c_length, MCL_LONG_INT, c_tag, c_source)
    if (present(err)) err = error
end subroutine recv_int64_array3d


!> Receives single-precision floating point.
subroutine recv_float32(buffer, length, tag, source, err)
    use, intrinsic :: iso_fortran_env, only: real32
    use iso_c_binding, only: c_ptr, c_int, c_loc
    !> Pointer to the buffer to receive data.
    real(kind=real32), target :: buffer
    !> Size of data buffer.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program to receive data from.
    integer :: source
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_source
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_source = source

    error = c_mcl_receive(c_buffer, c_length, MCL_FLOAT, c_tag, c_source)
    if (present(err)) err = error
end subroutine recv_float32

!> Receives 1-dimensional single-precision floating point array.
subroutine recv_float32_array1d(buffer, length, tag, source, err)
    use, intrinsic :: iso_fortran_env, only: real32
    use iso_c_binding, only: c_ptr, c_int, c_loc
    !> Pointer to the buffer to receive data.
    real(kind=real32), dimension(:), target :: buffer
    !> Size of data buffer.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program to receive data from.
    integer :: source
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_source
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_source = source

    error = c_mcl_receive(c_buffer, c_length, MCL_FLOAT, c_tag, c_source)
    if (present(err)) err = error
end subroutine recv_float32_array1d

!> Receives 2-dimensional single-precision floating point array.
subroutine recv_float32_array2d(buffer, length, tag, source, err)
    use, intrinsic :: iso_fortran_env, only: real32
    use iso_c_binding, only: c_ptr, c_int, c_loc
    !> Pointer to the buffer to receive data.
    real(kind=real32), dimension(:,:), target :: buffer
    !> Size of data buffer.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program to receive data from.
    integer :: source
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_source
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_source = source

    error = c_mcl_receive(c_buffer, c_length, MCL_FLOAT, c_tag, c_source)
    if (present(err)) err = error
end subroutine recv_float32_array2d

!> Receives 3-dimensional single-precision floating point array.
subroutine recv_float32_array3d(buffer, length, tag, source, err)
    use, intrinsic :: iso_fortran_env, only: real32
    use iso_c_binding, only: c_ptr, c_int, c_loc
    !> Pointer to the buffer to receive data.
    real(kind=real32), dimension(:,:,:), target :: buffer
    !> Size of data buffer.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program to receive data from.
    integer :: source
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_source
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_source = source

    error = c_mcl_receive(c_buffer, c_length, MCL_FLOAT, c_tag, c_source)
    if (present(err)) err = error
end subroutine recv_float32_array3d


!> Receives double-precision floating point.
subroutine recv_float64(buffer, length, tag, source, err)
    use, intrinsic :: iso_fortran_env, only: real64
    use iso_c_binding, only: c_ptr, c_int, c_loc
    !> Pointer to the buffer to receive data.
    real(kind=real64), target :: buffer
    !> Size of data buffer.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program to receive data from.
    integer :: source
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_source
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_source = source

    error = c_mcl_receive(c_buffer, c_length, MCL_DOUBLE, c_tag, c_source)
    if (present(err)) err = error
end subroutine recv_float64

!> Receives 1-dimensional double-precision floating point array.
subroutine recv_float64_array1d(buffer, length, tag, source, err)
    use, intrinsic :: iso_fortran_env, only: real64
    use iso_c_binding, only: c_ptr, c_int, c_loc
    !> Pointer to the buffer to receive data.
    real(kind=real64), dimension(:), target :: buffer
    !> Size of data buffer.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program to receive data from.
    integer :: source
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_source
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_source = source

    error = c_mcl_receive(c_buffer, c_length, MCL_DOUBLE, c_tag, c_source)
    if (present(err)) err = error
end subroutine recv_float64_array1d

!> Receives 2-dimensional double-precision floating point array.
subroutine recv_float64_array2d(buffer, length, tag, source, err)
    use, intrinsic :: iso_fortran_env, only: real64
    use iso_c_binding, only: c_ptr, c_int, c_loc
    !> Pointer to the buffer to receive data.
    real(kind=real64), dimension(:,:), target :: buffer
    !> Size of data buffer.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program to receive data from.
    integer :: source
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_source
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_source = source

    error = c_mcl_receive(c_buffer, c_length, MCL_DOUBLE, c_tag, c_source)
    if (present(err)) err = error
end subroutine recv_float64_array2d

!> Receives 3-dimensional double-precision floating point array.
subroutine recv_float64_array3d(buffer, length, tag, source, err)
    use, intrinsic :: iso_fortran_env, only: real64
    use iso_c_binding, only: c_ptr, c_int, c_loc
    !> Pointer to the buffer to receive data.
    real(kind=real64), dimension(:,:,:), target :: buffer
    !> Size of data buffer.
    integer :: length
    !> Message tag.
    integer :: tag
    !> ID of the program to receive data from.
    integer :: source
    !> Error status.
    integer, optional, intent(out) :: err

    type(c_ptr) :: c_buffer
    integer(kind=c_int) :: c_length
    integer(kind=c_int) :: c_tag
    integer(kind=c_int) :: c_source
    integer(kind=c_int) :: error

    c_buffer = c_loc(buffer)
    c_length = length
    c_tag = tag
    c_source = source

    error = c_mcl_receive(c_buffer, c_length, MCL_DOUBLE, c_tag, c_source)
    if (present(err)) err = error
end subroutine recv_float64_array3d

end module mcl
