//    MCL: MiMiC Communication Library
//    Copyright (C) 2015-2025  The MiMiC Authors (see CONTRIBUTORS file for details).
//
//    This file is part of MCL.
//
//    MCL is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as
//    published by the Free Software Foundation, either version 3 of
//    the License, or (at your option) any later version.
//
//    MCL is distributed in the hope that it will be useful, but
//    WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.

#ifndef MCL_ENDPOINT_H_
#define MCL_ENDPOINT_H_

#include <memory>
#include <vector>

#include "transfer_logger.h"
#include "transport.h"

namespace mcl {

/// Represents endpoints (programs) involved in data communication.
class Endpoint {

  public:
    explicit Endpoint(std::shared_ptr<Transport> protocol) : protocol_(protocol), id_(protocol->program_id()) { };
    ~Endpoint() { }

    /// Initializes the endpoint and performs a communication test.
    ///   \return Error code.
    int Initialize();

    /// Destroys the endpoint.
    ///   \return Error code.
    int Finalize();

    /// Sends data to the specified endpoint.
    ///   \param[in]  data         Pointer to the array containing data to be sent.
    ///   \param[in]  length       Number of elements in the data array.
    ///   \param[in]  data_type    Type of the data to be sent.
    ///   \param[in]  tag          Message tag used for matching.
    ///   \param[in]  destination  ID of the receiving endpoint.
    ///   \return Error code
    int Send(void *data, int length, int data_type, int tag, int destination);

    /// Receives data from the specified endpoint.
    ///   \param[out] data       Pointer to the array to serve as buffer for the data to be received.
    ///   \param[in]  length     Number of elements in the data array.
    ///   \param[in]  data_type  Type of the data to be received.
    ///   \param[in]  tag        Message tag used for matching.
    ///   \param[in]  source     ID of the sending endpoint.
    ///   \return Error code
    int Receive(void *data, int length, int data_type, int tag, int source);

    /// Returns the size of the message waiting in the queue (analogue to MPI_Probe).
    ///   \param[in]  data_type  Type of data to check.
    ///   \param[in]  source     ID of the sending program.
    ///   \return Size of the incoming message.
    int ProbeSize(int data_type, int source);

    int id() const { return id_; }
    void set_id(int id) { id_ = id; }
    const std::vector<int> &location_ids() const { return location_ids_; }


  protected:
    std::shared_ptr<Transport> protocol_;   ///< Transport protocol used to send data.
    int id_;                                ///< Endpoint ID.
    std::vector<int> location_ids_;         ///< List of known destinations/sources for a given endpoint.
    bool is_test_run_;                      ///< Runs in a test mode, with data read from a file.
    bool is_recorded_;                      ///< Records all transferred data in a file.
    bool skip_send_record_;                 ///< Skips recording send transfers.
    TransferLogger *transfer_log_;          ///< Log for recording data transfers.

};

} // namespace mcl

#endif // MCL_ENDPOINT_H_
