//    MCL: MiMiC Communication Library
//    Copyright (C) 2015-2025  The MiMiC Authors (see CONTRIBUTORS file for details).
//
//    This file is part of MCL.
//
//    MCL is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as
//    published by the Free Software Foundation, either version 3 of
//    the License, or (at your option) any later version.
//
//    MCL is distributed in the hope that it will be useful, but
//    WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.

#include "env_controls.h"

#include <cctype>
#include <cstdlib>
#include <string>
#include <typeinfo>

#include "error_print.h"

namespace mcl {

template <class T>
bool EnvVariable<T>::IsDefined() {
    if (!found_) {
        PrintErrorMessage("The required environment variable, " + AddQuotes(name_) + " is not assigned!",
                          __FILE__, __LINE__);
        return false;
    } else if (!valid_) {
        std::string message = "The environment variable, " + AddQuotes(name_) + " is present but invalid!\n";
        if (typeid(T) == typeid(int)) message += "Must be an integer.";
        if (typeid(T) == typeid(bool)) message += "Must be one of the following: ON/OFF, TRUE/FALSE, 0/1";
        PrintErrorMessage(message, __FILE__, __LINE__);
        return false;
    }
    return true;
}

template class EnvVariable<std::string>;
template class EnvVariable<int>;
template class EnvVariable<bool>;

std::string GetString(const char *var_name) {
    const char *value = std::getenv(var_name);
    if (value != nullptr) {
        return std::string{value};
    } else {
        return std::string();
    }
}
    
// Returns true if string represents an integer.
bool RepresentsInteger(const std::string &input) {
    if (input.empty()) return false;

    // integers may be negative
    const size_t minus_pos = input.find("-");
    auto iter = input.cbegin();
    if (minus_pos != std::string::npos) {
        if (minus_pos == 0 && input.size() > 1) {
            ++iter;
        } else {
            // either the var is just a string with one char, '-'
            // or it's a string with a minus somewhere
            return false;
        }
    }

    // iterate through the string to find a non-integer character
    for (; iter != input.cend(); ++iter) {
        if (!std::isdigit(static_cast<int>(*iter))) {
            return false;
        }
    }
    return true;
}
    
void EnvVariableString::ReadValue() {
    found_ = false;
    std::string value = GetString(name_);
    if (!value.empty()) {
        value_ = value;
        found_ = true;
    }
}

void EnvVariableInt::ReadValue() {
    found_ = false;
    valid_ = true;
    std::string value = GetString(name_);
    if (!value.empty()) {
        found_ = true;
        valid_ = RepresentsInteger(value);
        if (valid_) value_ = std::stoi(value);
    }
}

void EnvVariableBool::ReadValue() {
    found_ = false;
    valid_ = true;
    std::string value = GetString(name_);
    if (!value.empty()) {
        found_ = true;
        valid_ = false;
        if (value == "ON" || value == "TRUE" || value == "1") {
            value_ = true;
            valid_ = true;
        } else if (value == "OFF" || value == "FALSE" || value == "0") {
            value_ =  false;
            valid_ = true;
        }
    }
}

} // namespace mcl
