//    MCL: MiMiC Communication Library
//    Copyright (C) 2015-2025  The MiMiC Authors (see CONTRIBUTORS file for details).
//
//    This file is part of MCL.
//
//    MCL is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as
//    published by the Free Software Foundation, either version 3 of
//    the License, or (at your option) any later version.
//
//    MCL is distributed in the hope that it will be useful, but
//    WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.

#ifndef MCL_ENVCONTROLS_H_
#define MCL_ENVCONTROLS_H_

#include <string>

namespace mcl {

/// Abstract class storing an environment variable.
template <typename T>
class EnvVariable {

  public:
    EnvVariable(const char *name, T default_value) : name_(name), default_value_(default_value),
                                                     value_(default_value) { ReadValue(); };
    ~EnvVariable() = default;

    /// Reads the value of the environment variable.
    virtual void ReadValue() { }
    
    /// Returns a validity state of the variable and in the case of it is not being found/valid,
    /// prints an error message.
    ///   \return Variable is defined and its value is valid.
    ///   \note   Used for required variables, i.e., the ones that have to be specified by user.
    bool IsDefined();

    const char* name() { return name_; }
    bool found() { return found_; }
    T default_value() { return default_value_; }
    T value() { 
      ReadValue();
      return value_;
    }
    

  protected:
    const char *name_;        ///< Name of the environment variable.
    const T default_value_;   ///< Default value.
    T value_;                 ///< Variable value.
    bool found_ = false;      ///< Environment variable is defined.
    bool valid_ = true;       ///< Read environment variable is valid.

};

/// String environment variable.
class EnvVariableString : public EnvVariable<std::string> {
  public:
    EnvVariableString(const char *name, std::string default_value) : EnvVariable(name, default_value) { };
    void ReadValue() override;
};

/// Integer environment variable.
class EnvVariableInt : public EnvVariable<int> {
  public:
    EnvVariableInt(const char *name, int default_value) : EnvVariable(name, default_value) { };
    void ReadValue() override;
};

/// Boolean environment variable.
class EnvVariableBool : public EnvVariable<bool> {
  public:
    EnvVariableBool(const char *name, bool default_value) : EnvVariable(name, default_value) { };
    void ReadValue() override;
};


/// Key used to retrieve the communication mechanism environment variable.
static EnvVariableString env_communication_mode = EnvVariableString("MCL_COMM_MODE", "MPI_MPMD");

/// Key used to specify directory in which port files will be written (for MpiPortTransport).
static EnvVariableString env_port_directory = EnvVariableString("MCL_PORT_DIR", "");

/// File from which the data is read for a test transport run (for StubTransport).
static EnvVariableString env_test_data_file = EnvVariableString("MCL_TEST_DATA", "");

/// Key used to assign program number (for MpiPortTransport and StubTransport).
static EnvVariableInt env_program_number = EnvVariableInt("MCL_PROGRAM", -1);

/// Key used to set timeout for opening ports (for MpiPortTransport).
static EnvVariableInt env_timeout_value = EnvVariableInt("MCL_TIMEOUT", 5000);

/// Key used to turn on recording of the communication between MCL
/// and an external program (for StubTransport).
static EnvVariableBool env_record_communication = EnvVariableBool("MCL_RECORD_COMMUNICATION", false);

/// Key used to discard the sent data when recording of the communication
/// between MCL and an external program (for StubTransport).
static EnvVariableBool env_record_discard_send = EnvVariableBool("MCL_RECORD_DISCARD_SEND", false);

} // namespace mcl

#endif // MCL_ENVCONTROLS_H_
