//    MCL: MiMiC Communication Library
//    Copyright (C) 2015-2025  The MiMiC Authors (see CONTRIBUTORS file for details).
//
//    This file is part of MCL.
//
//    MCL is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as
//    published by the Free Software Foundation, either version 3 of
//    the License, or (at your option) any later version.
//
//    MCL is distributed in the hope that it will be useful, but
//    WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.

#include "error_print.h"

#include <filesystem>
#include <iostream>
#include <regex>
#include <string>

namespace mcl {

const std::string kIndent("  ");
const int  kLineWidth = 85;

// Applies indentation for multiline messages.
static std::string ApplyIndents(const std::string &message) {
    return std::regex_replace(message, std::regex("\n"), "\n" + kIndent);
}

// Parses out the file name from a path.
static std::string GetFileName(const std::string &path) {
    namespace fs = std::filesystem;
    return fs::path(path).filename().string();
}

// Returns a delimiter for an error block.
static std::string SectionDelimiter(const std::string &section_name) {
    std::string delimiter(kLineWidth, '~');
    int len = section_name.length();
    if ((len > 0) && (len < kLineWidth - 33))
        delimiter.replace(3, len+2, " " + section_name + " ");
    return delimiter + '\n';
}

void PrintErrorMessage(const std::string &message, const char *file, int line, int error_code) {
    std::cerr << SectionDelimiter("MCL ~ ERROR");
    std::cerr << kIndent << ApplyIndents(message) << std::endl; 
    if (error_code != 0)
        std::cerr << kIndent << " > Additional MPI error code: " << error_code << std::endl;
    std::cerr << kIndent << " > Called in " << AddQuotes(GetFileName(file)) << " on the line no. " << line << std::endl;
    std::cerr << SectionDelimiter(""); 
}

void PrintWarning(const std::string &message) {
    std::string multiline_message = std::regex_replace(message, std::regex("\n"), "\n" + kIndent);
    std::cerr << SectionDelimiter("MCL ~ WARNING"); 
    std::cerr << kIndent << multiline_message << std::endl; 
    std::cerr << SectionDelimiter(""); 
}

std::string AddQuotes(const std::string &message) {
  return '\"' + message + '\"';
}

std::string AddQuotes(const char *message) {
  return AddQuotes(std::string(message));
}

} // namespace mcl
