//    MCL: MiMiC Communication Library
//    Copyright (C) 2015-2025  The MiMiC Authors (see CONTRIBUTORS file for details).
//
//    This file is part of MCL.
//
//    MCL is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as
//    published by the Free Software Foundation, either version 3 of
//    the License, or (at your option) any later version.
//
//    MCL is distributed in the hope that it will be useful, but
//    WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.

#include "main_class.h"

#include <string>

#include "endpoint.h"
#include "env_controls.h"
#include "error_codes.h"
#include "error_print.h"
#include "transport/mpi_mpmd_transport.h"
#include "transport/mpi_port_transport.h"
#include "transport/stub_transport.h"

namespace mcl {

int MclMain::Initialize(void *args) {
    if (is_initialized_) {
        PrintErrorMessage("MCL_Initialized is called, but MCL is already initialized!", __FILE__, __LINE__);
        return Abort(kMclAlreadyInit);
    }

    // get the environment variable for selecting the mode of communication
    std::string communication_key = env_communication_mode.value();
        
    if (protocol_ == nullptr) {
        // select the communication mechanism
        if (communication_key == "MPI_PORT") {
            // MPI mechanism in which the programs are launched independently
            protocol_ = std::make_shared<MpiPortTransport>(args);
        } else if (communication_key == "MPI_MPMD") {
            // MPI mechanism in which MPI_COMM_WORLD is split
            protocol_ = std::make_shared<MpiMpmdTransport>(args);
        } else if (communication_key == "TEST_STUB") {
            // mode for external program testing
            protocol_ = std::make_shared<StubTransport>();
        } else {
            PrintWarning("Unknown communication mechanism " + AddQuotes(communication_key) + " was chosen! \n" +
                         "MPI_MPMD will be used!");
            protocol_ = std::make_shared<MpiMpmdTransport>(args);
        }
    }


    int result;
    result = protocol_->Initialize(args);
    if (result != kSuccess)
        return protocol_->Abort(result);

    endpoint_ = nullptr;
    if (protocol_->is_server() || protocol_->is_client()) {
        endpoint_ = std::make_shared<Endpoint>(protocol_);
        result = endpoint_->Initialize();
        if (result != kSuccess)
            return protocol_->Abort(result);
    }
    
    is_initialized_ = true;

    return result;
}

int MclMain::Finalize() {
    if (!is_initialized_) return Abort(kMclNotInit);

    is_initialized_ = false;

    int result;
    if (endpoint_ != nullptr) {
        result = endpoint_->Finalize();
        if (result != kSuccess) return result;
    }
    
    result = protocol_->Finalize();
    if (result != kSuccess) return result;
    protocol_ = nullptr;

    return kSuccess;
}

int MclMain::Abort(int error_code) {
    if (!is_initialized_) { 
        PrintErrorMessage("It seems that MCL was not initialized properly.\n"
                          "Please call MCL_Initialize first!", __FILE__, __LINE__);
        std::abort();
    }
    return protocol_->Abort(error_code);
}

int MclMain::Send(void *data, int length, int data_type, int tag, int destination) {
    if (!is_initialized_) return Abort(kMclNotInit);
    if (endpoint_ != nullptr)
        return endpoint_->Send(data, length, data_type, tag, destination);
    return kSuccess;
}

int MclMain::Receive(void *data, int length, int data_type, int tag, int source) {
    if (!is_initialized_) return Abort(kMclNotInit);
    if (endpoint_ != nullptr)
        return endpoint_->Receive(data, length, data_type, tag, source);
    return kSuccess;
}

int MclMain::ProbeSize(int data_type, int source) {
    if (!is_initialized_) return Abort(kMclNotInit);
    if (endpoint_ != nullptr)
        return endpoint_->ProbeSize(data_type, source);
    return 0;
}

int MclMain::num_programs() {
    if (!is_initialized_) return Abort(kMclNotInit);
    return protocol_->num_programs();
}

int MclMain::program_id() {
    if (!is_initialized_) return Abort(kMclNotInit);
    return protocol_->program_id();
}

} // namespace mcl
