//    MCL: MiMiC Communication Library
//    Copyright (C) 2015-2025  The MiMiC Authors (see CONTRIBUTORS file for details).
//
//    This file is part of MCL.
//
//    MCL is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as
//    published by the Free Software Foundation, either version 3 of
//    the License, or (at your option) any later version.
//
//    MCL is distributed in the hope that it will be useful, but
//    WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.

#ifndef MCL_MAINCLASS_H_
#define MCL_MAINCLASS_H_

#include <memory>

#include "endpoint.h"

namespace mcl {

/// Singleton main class of the MCL.
class MclMain {

  public:
    void operator=(MclMain const&) = delete;

    /// Retrieves or creates the singleton instance.
    static MclMain &GetInstance() {
        static MclMain instance;
        return instance;
    }

    /// Initializes clients and server for receiving/sending data.
    ///   \param[in,out] args  Arguments depend on the chosen transport type.
    ///   \return Error code.
    int Initialize(void *args);

    /// Destroys the endpoint.
    ///   \return Error code
    int Finalize();

    /// Terminates all processes.
    ///   \param[in]  error_code  Error code to return.
    ///   \return Error code.
    int Abort(int error_code);

    /// Sends data to the specified program.
    ///   \param[in]  data         Pointer to the array containing data to be sent.
    ///   \param[in]  length       Number of elements in the data array.
    ///   \param[in]  data_type    Type of the data to be sent.
    ///   \param[in]  tag          Message tag used for matching.
    ///   \param[in]  destination  ID of the receiving program.
    ///   \return Error code.
    int Send(void *data, int length, int data_type, int tag, int destination);

    /// Receives data from the specified program.
    ///   \param[out] data       Pointer to the array to serve as buffer for the data to be received.
    ///   \param[in]  length     Number of elements in the data array.
    ///   \param[in]  data_type  Type of the data to be received.
    ///   \param[in]  tag        Message tag used for matching.
    ///   \param[in]  source     ID of the sending program.
    ///   \return Error code.
    int Receive(void *data, int length, int data_type, int tag, int source);

    /// Returns the size of the message waiting in the queue (analogue to MPI_Probe).
    ///   \param[in]  data_type  Type of data to check.
    ///   \param[in]  source     ID of the sending program.
    ///   \return Size of the incoming message.
    int ProbeSize(int data_type, int source);

    int num_programs();
    int program_id();
    bool is_initialized() { return is_initialized_; };
    const std::shared_ptr<Endpoint> &endpoint() const { return endpoint_; }
    void set_protocol(std::shared_ptr<Transport> protocol) { protocol_ = protocol; }


  private:
    MclMain() = default;

    std::shared_ptr<Transport> protocol_ = nullptr;   ///< Transport protocol.
    std::shared_ptr<Endpoint> endpoint_ = nullptr;    ///< Endpoint for a given program.
    bool is_initialized_ = false;                     ///< Initialization status.

};

} // namespace mcl

#endif // MCL_MAINCLASS_H_
