//    MCL: MiMiC Communication Library
//    Copyright (C) 2015-2025  The MiMiC Authors (see CONTRIBUTORS file for details).
//
//    This file is part of MCL.
//
//    MCL is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as
//    published by the Free Software Foundation, either version 3 of
//    the License, or (at your option) any later version.
//
//    MCL is distributed in the hope that it will be useful, but
//    WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.

#ifndef MCL_TRANSPORT_TRANSPORT_H_
#define MCL_TRANSPORT_TRANSPORT_H_

namespace mcl {

/// Abstract class of the transport layer.
class Transport {

  public:
    Transport() = default;
    virtual ~Transport() = default;

    /// Initializes clients and server for receiving/sending data.
    ///   \param[in,out] args  Arguments depend on the chosen transport type.
    ///   \return Error code.
    virtual int Initialize(void *args) = 0;

    /// Destroys the port associated with the specified path.
    ///   \return Error code.
    virtual int Finalize() = 0;

    /// Terminates all processes.
    ///   \param[in] error_code  Error code to return.
    ///   \return Error code.
    virtual int Abort(int error_code) = 0;

    /// Sends data to the specified program.
    ///   \param[in] data         Pointer to the array containing data to be sent.
    ///   \param[in] length       Number of elements in the data array.
    ///   \param[in] data_type    Type of the data to be sent.
    ///   \param[in] tag          Message tag used for matching.
    ///   \param[in] destination  ID of the receiving program.
    ///   \return Error code.
    virtual int Send(void *data, int length, int data_type, int tag, int destination) = 0;

    /// Receives data from the specified program.
    ///   \param[out] data       Pointer to the array to serve as buffer for the data to be received.
    ///   \param[in]  length     Number of elements in the data array.
    ///   \param[in]  data_type  Type of the data to be received.
    ///   \param[in]  tag        Message tag used for matching.
    ///   \param[in]  source     ID of the sending program.
    ///   \return Error code.
    virtual int Receive(void *data, int length, int data_type, int tag, int source) = 0;

    /// Returns the size of the message waiting in the queue (analogue to MPI_Probe).
    ///   \param[in]  data_type  Type of data to be received.
    ///   \param[in]  source     ID of the sending program.
    ///   \return Size of the incoming message.
    virtual int ProbeSize(int data_type, int source) = 0;

    virtual bool is_server() const { return is_server_; };
    virtual bool is_client() const { return is_client_; };
    virtual int num_programs() const { return num_programs_; };
    virtual int program_id() const { return program_id_; };


  protected:
    int num_programs_;  ///< Number of programs (clients + server).
    int program_id_;    ///< ID of the program run with this rank.
    bool is_server_;    ///< This rank is an MCL server.
    bool is_client_;    ///< This rank is an MCL client.

};

} // namespace mcl

#endif // MCL_TRANSPORT_TRANSPORT_H_
