! Copyright (C) 2022  Light and Molecules Group

! This program is free software: you can redistribute it and/or modify
! it under the terms of the GNU General Public License as published by
! the Free Software Foundation, either version 3 of the License, or
! (at your option) any later version.

! This program is distributed in the hope that it will be useful,
! but WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
! GNU General Public License for more details.

! You should have received a copy of the GNU General Public License
! along with this program.  If not, see <https://www.gnu.org/licenses/>.

module mod_print_utils
  use mod_kinds, only: dp
  use mod_constants, only: MAX_STR_SIZE
  implicit none

  private

  interface message
     module procedure print_message
     module procedure print_message_d1
     module procedure print_mat_r64_d1
     module procedure print_mat_r64_d2
     module procedure print_mat_c64_d1
     module procedure print_mat_i32_d1
     module procedure print_mat_i32_d2
  end interface message
  public :: message
  public :: log_file

  ! String manipulation
  public :: to_upper, to_lower
  public :: pad_str

  public :: date

contains

  function to_upper(strIn) result(strOut)
    !! Convert a string to upper case.
    !!
    !! Adapted from [there] (http://www.star.le.ac.uk/~cgp/fortran.html) (25 May 2012)
    !! Original author: Clive Page
    character(len=*), intent(in) :: strIn
    character(len=len(strIn)) :: strOut
    integer :: i,j

    do i = 1, len(strIn)
       j = iachar(strIn(i:i))
       if (j>= iachar("a") .and. j<=iachar("z") ) then
          strOut(i:i) = achar(iachar(strIn(i:i))-32)
       else
          strOut(i:i) = strIn(i:i)
       end if
    end do

  end function to_upper


  function to_lower(strIn) result(strOut)
    !! Convert a string to lower case.
    !!
    !! Adapted from [there] (http://www.star.le.ac.uk/~cgp/fortran.html) (25 May 2012)
    !! Original author: Clive Page
    character(len=*), intent(in) :: strIn
    character(len=len(strIn)) :: strOut
    integer :: i,j

    do i = 1, len(strIn)
       j = iachar(strIn(i:i))
       if (j>= iachar("A") .and. j<=iachar("Z") ) then
          strOut(i:i) = achar(iachar(strIn(i:i))+32)
       else
          strOut(i:i) = strIn(i:i)
       end if
    end do

  end function to_lower


  pure function pad_str(str, total_length) result(res)
    character(len=*), intent(in) :: str
    integer, intent(in) :: total_length

    character(len=total_length) :: res

    integer :: len, pad

    len = len_trim(str)
    pad = int(total_length - len) / 2
    res = repeat(' ', pad)//trim(str)//repeat(' ', pad)
  end function pad_str


  function date()
    !! Format the current date: ``yyyy/mm/dd hh:mm:ss``.
    !!
    character(len=19) :: date

    ! character(len=10) :: b(3)
    character(len=64) :: fmt
    integer, dimension(8) :: values

    call date_and_time(values=values)
    write(fmt, '(A)') &
         & '(I4,"/",I2.2,"/",I2.2," ",I2.2,":",I2.2,":",I2.2)'

    write(date, fmt=fmt) values(1), values(2), values(3), values(5),&
         & values(6), values(7)
  end function date


  function format_string(str, prefix, time_stamp, file, proc)
    !! Format a string for logging.
    !!
    !! This function will convert ``str`` in the following form:
    !!
    !!    ``[prefix]<date>(In file, proc) str``
    !!
    !! - ``time_stamp`` controls the presence of ``date`` (generated by the ``date``
    !!   function) ;
    !! - if ``prefix`` is not present the ``[prefix]`` part is omitted ;
    !! - if ``file`` or ``proc`` is not present the corresponding part is omitted.
    character(len=*), intent(in), optional :: str
    !! Message to print.
    character(len=*), intent(in), optional :: prefix
    !! String with which to prefix the message (will be between
    !! ``[]``, default: empty).
    logical, intent(in), optional :: time_stamp
    !! Add a time-stamp to the string (generated by the function
    !! ``date()`` from this module, default: ``.true.``).
    character(len=*), intent(in), optional :: file
    !! File in which the routine is called (default: empty).
    character(len=*), intent(in), optional :: proc
    !! Name of the procedure calling this routine (default: empty).

    character(len=:), allocatable :: format_string
    character(len=:), allocatable :: to_print

    ! character(len=1024) :: to_print, fstring
    character(len=1024) :: fstring
    logical :: tstamp

    to_print = ''
    if (present(prefix)) to_print = '['//trim(prefix)//']'

    tstamp = .true.
    if (present(time_stamp))  tstamp = time_stamp
    if (tstamp) then
       to_print = trim(to_print)//'<'//date()//'>'
    end if

    if ((present(file)) .or. present(proc)) then

       fstring = '(In'
       if (present(file)) then

          if (file /= '') then
             fstring = trim(fstring)//' '//trim(file)
             if (present(proc)) then
                fstring = trim(fstring)//', '//trim(proc)//')'
             else
                fstring = trim(fstring)//')'
             end if
          end if

       else
          if (proc /= '') then
             fstring = trim(fstring)//' '//trim(proc)//')'
          end if
       end if

       if (fstring /= '(In') then
          to_print = trim(to_print)//' '//trim(fstring)
       end if
    end if

    if (present(str)) then
       if (to_print /= '') then
          to_print = trim(to_print)//' '//trim(str)
       else
          to_print = trim(str)
       end if
    end if

    format_string = trim(to_print)
  end function format_string


  subroutine log_file(filename, unit, prefix, time_stamp, title, file, proc)
    !! Print the content of the given file.
    !!
    !! The unit should be opened before printing the message.
    !! Optionnally, it is possible to prefix ``msg`` with a
    !! ``prefix`` and to add a time stamp, as well as some
    !! information about the location of the print with ``file``
    !! and ``proc`` information.
    character(len=*), intent(in) :: filename
    !! File to print.
    integer, intent(in) :: unit
    !! (Opened) File unit where to print the message.
    character(len=*), intent(in), optional :: prefix
    !! String with which to prefix the message (will be between
    !! ``[]``, default: empty).
    logical, intent(in), optional :: time_stamp
    !! Add a time-stamp to the string (generated by the function
    !! ``date()`` from this module, default: ``.true.``).
    character(len=*), intent(in), optional :: title
    !! Title to give to the file.
    character(len=*), intent(in), optional :: file
    !! File in which the routine is called (default: empty).
    character(len=*), intent(in), optional :: proc
    !! Name of the procedure calling this routine (default: empty).

    integer :: u, ierr
    character(len=MAX_STR_SIZE) :: buf

    write(unit, '(A)') &
         & trim(format_string(&
         & str=title, prefix=prefix, time_stamp=time_stamp, file=file, proc=proc&
         &))

    open(newunit=u, file=filename, action='read')
    do
       read(u, '(A)', iostat=ierr) buf
       if (ierr /= 0) exit

       write(unit, '(A)') trim(buf)
    end do
    close(u)

  end subroutine log_file


  subroutine print_message(msg, unit, prefix, time_stamp, file, proc)
    !! Print a message to the specified ``unit``.
    !!
    !! The unit should be opened before printing the message.
    !! Optionnally, it is possible to prefix ``msg`` with a
    !! ``prefix`` and to add a time stamp, as well as some
    !! information about the location of the print with ``file``
    !! and ``proc`` information.
    !!
    !! The message is then printed in the following format:
    !!
    !! [prefix]<time-stamp> (In file, proc) message
    !!
    !! The routine is intended to be used with ``nx_log_t`` objects,
    !! and may be used to print general log info, or error messages.
    character(len=*), intent(in) :: msg
    !! Message to print.
    integer, intent(in) :: unit
    !! (Opened) File unit where to print the message.
    character(len=*), intent(in), optional :: prefix
    !! String with which to prefix the message (will be between
    !! ``[]``, default: empty).
    logical, intent(in), optional :: time_stamp
    !! Add a time-stamp to the string (generated by the function
    !! ``date()`` from this module, default: ``.true.``).
    character(len=*), intent(in), optional :: file
    !! File in which the routine is called (default: empty).
    character(len=*), intent(in), optional :: proc
    !! Name of the procedure calling this routine (default: empty).

    write(unit, '(A)') trim(format_string(&
         & str=msg, prefix=prefix, time_stamp=time_stamp, file=file, proc=proc&
         &))

  end subroutine print_message


  subroutine print_message_d1(msg, unit, prefix, nlines, time_stamp, title, file, proc)
    !! Print an array of strings to the specified ``unit``.
    !!
    !! The unit should be opened before printing the message.
    !! Optionnally, it is possible to prefix ``msg`` with a
    !! ``prefix`` and to add a time stamp, as well as some
    !! information about the location of the print with ``file``
    !! and ``proc`` information.
    !!
    !! The message is then printed in the following format:
    !!
    !! [prefix]<time-stamp> (In file, proc) message
    !!
    !! The routine is intended to be used with ``nx_log_t`` objects,
    !! and may be used to print general log info, or error messages.
    character(len=*), dimension(:), intent(in) :: msg
    !! Message to print.
    integer, intent(in) :: unit
    !! (Opened) File unit where to print the message.
    character(len=*), intent(in), optional :: prefix
    !! String with which to prefix the message (will be between
    !! ``[]``, default: empty).
    integer, intent(in), optional :: nlines
    !! Number of lines of the array to print.
    logical, intent(in), optional :: time_stamp
    !! Add a time-stamp to the string (generated by the function
    !! ``date()`` from this module, default: ``.true.``).
    character(len=*), intent(in), optional :: title
    !! Title of the array (default: empty).
    character(len=*), intent(in), optional :: file
    !! File in which the routine is called (default: empty).
    character(len=*), intent(in), optional :: proc
    !! Name of the procedure calling this routine (default: empty).

    character(len=MAX_STR_SIZE) :: to_print
    integer :: i, lines_to_print

    to_print = format_string(&
         & str=title, prefix=prefix, time_stamp=time_stamp, file=file, proc=proc&
         &)

    lines_to_print = size(msg)
    if (present(nlines)) lines_to_print = nlines

    if (to_print /= '') then
       write(unit, '(A)') trim(to_print)
    end if

    do i=1, lines_to_print
       write(unit, '(A)') trim(msg(i))
    end do

  end subroutine print_message_d1


  subroutine print_mat_r64_d1(data, unit, prefix, time_stamp, title, &
       & fmt, labels, fac, file, proc)
    !! Print the given vector to ``unit``.
    !!
    !! The unit should be opened before printing the message.  If
    !! ``prefix``, ``title``, ``file`` or ``proc`` is given, or if
    !! ``time_stamp`` is set to ``.true.``, then the vector is printed
    !! after a line:
    !!
    !! [prefix]<time-stamp> (In ``file``, ``proc``) title
    !!
    !! Else, the vector is printed following a blank line. It is
    !! possible to modify the printing format with ``fmt`` (default:
    !! ``F20.12``), and to scale the vector while printing it with
    !! the factor ``fac``.
    real(dp), dimension(:), intent(in) :: data
    !! Vector to print.
    integer, intent(in) :: unit
    !! (Opened) File unit where to print the message.
    character(len=*), intent(in), optional :: prefix
    !! String with which to prefix the message (will be between
    !! ``[]``, default: empty).
    logical, intent(in), optional :: time_stamp
    !! Add a time-stamp to the string (generated by the function
    !! ``date()`` from this module, default: ``.true.``).
    character(len=*), intent(in), optional :: title
    !! Title to print (default: empty).
    character(len=*), intent(in), optional :: fmt
    !! Format for printing (default: ``F20.12``).
    character(len=*), dimension(:), intent(in), optional :: labels
    !! Labels for the lines of the vector.
    real(dp), intent(in), optional :: fac
    !! Scaling factor for printing.
    character(len=*), intent(in), optional :: file
    !! File in which the routine is called (default: empty).
    character(len=*), intent(in), optional :: proc
    !! Name of the procedure calling this routine (default: empty).

    character(len=MAX_STR_SIZE) :: to_print
    character(len=MAX_STR_SIZE) :: prtfmt
    integer :: i

    to_print = format_string(&
         & str=title, prefix=prefix, time_stamp=time_stamp, file=file, proc=proc&
         &)

    write(unit, '(A)') trim(to_print)

    ! Format and write vector
    prtfmt = '(F20.12)'
    if (present(fmt)) prtfmt = fmt

    if (present(labels)) then
       prtfmt = '(A,'//trim(prtfmt)//')'
    else
       prtfmt = '('//trim(prtfmt)//')'
    end if

    do i=1, size(data)
       if (present(fac)) then
          if (present(labels)) then
             write(unit,  fmt=prtfmt) labels(i), fac*data(i)
          else
             write(unit, fmt=prtfmt) fac*data(i)
          end if
       else
          if (present(labels)) then
             write(unit,  fmt=prtfmt) labels(i), data(i)
          else
             write(unit, fmt=prtfmt) data(i)
          end if
       end if
    end do
  end subroutine print_mat_r64_d1


  subroutine print_mat_c64_d1(data, unit, prefix, time_stamp, title, &
       & fmt, labels, fac, file, proc)
    !! Print the given vector to ``unit``.
    !!
    !! The unit should be opened before printing the message.  If
    !! ``prefix``, ``title``, ``file`` or ``proc`` is given, or if
    !! ``time_stamp`` is set to ``.true.``, then the vector is printed
    !! after a line:
    !!
    !! [prefix]<time-stamp> (In ``file``, ``proc``) title
    !!
    !! Else, the vector is printed following a blank line. It is
    !! possible to modify the printing format with ``fmt`` (default:
    !! ``F20.12``), and to scale the vector while printing it with
    !! the factor ``fac``.
    complex(dp), dimension(:), intent(in) :: data
    !! Vector to print.
    integer, intent(in) :: unit
    !! (Opened) File unit where to print the message.
    character(len=*), intent(in), optional :: prefix
    !! String with which to prefix the message (will be between
    !! ``[]``, default: empty).
    logical, intent(in), optional :: time_stamp
    !! Add a time-stamp to the string (generated by the function
    !! ``date()`` from this module, default: ``.true.``).
    character(len=*), intent(in), optional :: title
    !! Title to print (default: empty).
    character(len=*), intent(in), optional :: fmt
    !! Format for printing (default: ``F20.12``).
    character(len=*), dimension(:), intent(in), optional :: labels
    !! Labels for the lines of the vector.
    real(dp), intent(in), optional :: fac
    !! Scaling factor for printing.
    character(len=*), intent(in), optional :: file
    !! File in which the routine is called (default: empty).
    character(len=*), intent(in), optional :: proc
    !! Name of the procedure calling this routine (default: empty).

    character(len=MAX_STR_SIZE) :: to_print
    character(len=MAX_STR_SIZE) :: prtfmt
    integer :: i

    to_print = format_string(&
         & str=title, prefix=prefix, time_stamp=time_stamp, file=file, proc=proc&
         &)

    write(unit, '(A)') trim(to_print)

    ! Format and write vector
    prtfmt = '(2F20.12)'
    ! if (present(fmt)) prtfmt = '(2'//trim(fmt)//')'

    if (present(fmt)) prtfmt = '2'//trim(fmt)

    if (present(labels)) then
       prtfmt = '(A,'//trim(prtfmt)//')'
    else
       prtfmt = '('//trim(prtfmt)//')'
    end if

    do i=1, size(data)
       if (present(fac)) then
          if (present(labels)) then
             write(unit, fmt=prtfmt) &
                  & labels(i), real(fac*data(i)), aimag(fac*data(i))
          else
             write(unit, fmt=prtfmt) &
                  & real(fac*data(i)), aimag(fac*data(i))
          end if
       else
          if (present(labels)) then
             write(unit, fmt=prtfmt) &
                  & labels(i), real(data(i)), aimag(data(i))
          else
             write(unit, fmt=prtfmt) real(data(i)), aimag(data(i))
          end if
       end if
    end do
  end subroutine print_mat_c64_d1


  subroutine print_mat_r64_d2(data, unit, prefix, time_stamp, title, &
       & fmt, fac, expand, transpose, clabels, file, proc)
    !! Print the given matrix to ``unit``.
    !!
    !! The unit should be opened before printing the message.  If
    !! ``prefix``, ``title``, ``file`` or ``proc`` is given, or if
    !! ``time_stamp`` is set to ``.true.``, then the vector is printed
    !! after a line:
    !!
    !! [prefix]<time-stamp> (In ``file``, ``proc``) title
    !!
    !! Else, the matrix is printed following a blank line. It is
    !! possible to modify the printing format with ``fmt`` (default:
    !! ``F20.12``), and to scale the matrix while printing it with
    !! the factor ``fac``.  By default the matrix is printed as
    !! blocks of 4 columns, each block being labeled by its column
    !! number. If ``expand`` is set to ``.true.`` then the matrix is
    !! printed in full form.  It is also possible to print the
    !! transpose of the given matrix by setting ``transpose`` to
    !! ``.true.``.
    real(dp), dimension(:, :), intent(in) :: data
    !! Vector to print.
    integer, intent(in) :: unit
    !! (Opened) File unit where to print the message.
    character(len=*), intent(in), optional :: prefix
    !! String with which to prefix the message (will be between
    !! ``[]``, default: empty).
    logical, intent(in), optional :: time_stamp
    !! Add a time-stamp to the string (generated by the function
    !! ``date()`` from this module, default: ``.true.``).
    character(len=*), intent(in), optional :: title
    !! Title to print (default: empty).
    character(len=*), intent(in), optional :: fmt
    !! Format for printing (default: ``F20.12``).
    real(dp), intent(in), optional :: fac
    !! Scaling factor for printing.
    logical, intent(in), optional :: expand
    !! Expand the matrix in full form (default: ``.false.``)
    logical, intent(in), optional :: transpose
    !! Transpose the matrix for printing (default: ``.false.``)
    character(len=*), intent(in), dimension(:), optional :: clabels
    !! Labels for Matrix columns.
    character(len=*), intent(in), optional :: file
    !! File in which the routine is called (default: empty).
    character(len=*), intent(in), optional :: proc
    !! Name of the procedure calling this routine (default: empty).

    character(len=MAX_STR_SIZE) :: to_print
    character(len=MAX_STR_SIZE) :: prtfmt
    integer :: nrows, ncols, bb, blocks, rem, ii
    logical :: texp, ttrans
    character(len=MAX_STR_SIZE) :: number_fmt
    real(dp) :: tfac
    integer :: i, j

    to_print = format_string(&
         & str=title, prefix=prefix, time_stamp=time_stamp, file=file, proc=proc&
         &)

    write(unit, '(A)') trim(to_print)

    nrows = size(data, 1)
    ncols = size(data, 2)

    number_fmt = 'F20.12'
    if (present(fmt)) number_fmt = fmt

    texp = .false.
    if (present(expand)) texp = expand
    ttrans = .false.
    if (present(transpose)) ttrans = transpose
    tfac = 1.0_dp
    if (present(fac)) tfac = fac

    ! Set the formatting
    if (texp) then
       if (ttrans) then
          write(prtfmt, '(A,I0,A,A)') '(', ncols, trim(number_fmt), ')'
       else
          write(prtfmt, '(A,I0,A,A)') '(', nrows, trim(number_fmt), ')'
       end if
    else
       write(prtfmt, '(A,A,A)') '(4', trim(number_fmt), ')'
    end if

    if (.not. texp) then
       if (ttrans) then
          nrows = size(data, 2)
          ncols = size(data, 1)
       else
          nrows = size(data, 1)
          ncols = size(data, 2)
       end if

       blocks = int(ncols / 4)
       rem = modulo(ncols, 4)

       ii = 1
       do bb=1, blocks
          if ((blocks > 1) .and. (rem > 0)) then
             if (present(clabels)) then
                write(unit, '(4A)') (adjustr(clabels(ii+i)), i=0, 3)
             else
                write(unit, '(4I20)') (ii+i, i=0, 3)
             end if
          else
             if (present(clabels)) then
                write(unit, '(4A)') (adjustr(clabels(ii+i)), i=0, 3)
             end if
          end if

          do j=1, nrows
             if (ttrans) then
                write(unit, fmt=prtfmt) (tfac*data(ii+i, j), i=0, 3)
             else
                write(unit, fmt=prtfmt) (tfac*data(j, ii+i), i=0, 3)
             end if

          end do
          ii = ii + 4
       end do

       if (rem /= 0) then
          if ((blocks > 1) .and. (rem > 0)) then
             if (present(clabels)) then
                write(unit, '(4A)') (adjustr(clabels(ii+i)), i=0, 3)
             else
                write(unit, '(4I20)') (ii+i, i=0, 3)
             end if
          else
             if (present(clabels)) then
                write(unit, '(4A)') (adjustr(clabels(ii+i)), i=0, 3)
             end if
          end if

          do j=1, nrows
             if (ttrans) then
                write(unit, fmt=prtfmt) (tfac*data(ii+i, j), i=0, rem-1)
             else
                write(unit, fmt=prtfmt) (tfac*data(j, ii+i), i=0, rem-1)
             end if

          end do
       end if

    else
       if (ttrans) then
          do i=1, size(data, 2)
             write(unit, fmt=prtfmt) (tfac*data(j, i), j=1, size(data, 1))
          end do
       else
          do i=1, size(data, 1)
             write(unit, fmt=prtfmt) (tfac*data(i, j), j=1, size(data, 2))
          end do
       end if
    end if
  end subroutine print_mat_r64_d2


  subroutine print_mat_i32_d1(data, unit, prefix, time_stamp, title, &
       & fmt, fac, file, proc)
    !! Print the given integer vector to ``unit``.
    !!
    !! The unit should be opened before printing the message.  If
    !! ``prefix``, ``title``, ``file`` or ``proc`` is given, or if
    !! ``time_stamp`` is set to ``.true.``, then the vector is printed
    !! after a line:
    !!
    !! [prefix]<time-stamp> (In ``file``, ``proc``) title
    !!
    !! Else, the vector is printed following a blank line. It is
    !! possible to modify the printing format with ``fmt`` (default:
    !! ``I0``), and to scale the vector while printing it with
    !! the factor ``fac``.  If scaled, the vector will use a float
    !! format specifier, defaulting to ``F20.12``.
    integer, dimension(:), intent(in) :: data
    !! Vector to print.
    integer, intent(in) :: unit
    !! (Opened) File unit where to print the message.
    character(len=*), intent(in), optional :: prefix
    !! String with which to prefix the message (will be between
    !! ``[]``, default: empty).
    logical, intent(in), optional :: time_stamp
    !! Add a time-stamp to the string (generated by the function
    !! ``date()`` from this module, default: ``.true.``).
    character(len=*), intent(in), optional :: title
    !! Title to print (default: empty).
    character(len=*), intent(in), optional :: fmt
    !! Format for printing (default: ``F20.12``).
    real(dp), intent(in), optional :: fac
    !! Scaling factor for printing.
    character(len=*), intent(in), optional :: file
    !! File in which the routine is called (default: empty).
    character(len=*), intent(in), optional :: proc
    !! Name of the procedure calling this routine (default: empty).

    character(len=MAX_STR_SIZE) :: to_print
    character(len=MAX_STR_SIZE) :: prtfmt
    integer :: i

    to_print = format_string(&
         & str=title, prefix=prefix, time_stamp=time_stamp, file=file, proc=proc&
         &)

    write(unit, '(A)') trim(to_print)

    ! Format and write vector
    if (present(fac)) then
       prtfmt = '(F20.12)'
    else
       prtfmt = '(i0)'
    end if
    if (present(fmt)) prtfmt = '('//trim(fmt)//')'

    do i=1, size(data)
       if (present(fac)) then
          write(unit, fmt=prtfmt) fac*data(i)
       else
          write(unit, fmt=prtfmt) data(i)
       end if
    end do
  end subroutine print_mat_i32_d1


  subroutine print_mat_i32_d2(data, unit, prefix, time_stamp, title, &
       & fmt, fac, expand, transpose, clabels, file, proc)
    !! Print the given matrix to ``unit``.
    !!
    !! The unit should be opened before printing the message.  If
    !! ``prefix``, ``title``, ``file`` or ``proc`` is given, or if
    !! ``time_stamp`` is set to ``.true.``, then the vector is printed
    !! after a line:
    !!
    !! [prefix]<time-stamp> (In ``file``, ``proc``) title
    !!
    !! Else, the matrix is printed following a blank line. It is
    !! possible to modify the printing format with ``fmt`` (default:
    !! ``F20.12``), and to scale the matrix while printing it with
    !! the factor ``fac``.  By default the matrix is printed as
    !! blocks of 4 columns, each block being labeled by its column
    !! number. If ``expand`` is set to ``.true.`` then the matrix is
    !! printed in full form.  It is also possible to print the
    !! transpose of the given matrix by setting ``transpose`` to
    !! ``.true.``.
    !!
    !! If a scaling factor is present, the default format becomes
    !! ``F20.12``, and the elements of the matrix will be printed as floats.
    integer, dimension(:, :), intent(in) :: data
    !! Vector to print.
    integer, intent(in) :: unit
    !! (Opened) File unit where to print the message.
    character(len=*), intent(in), optional :: prefix
    !! String with which to prefix the message (will be between
    !! ``[]``, default: empty).
    logical, intent(in), optional :: time_stamp
    !! Add a time-stamp to the string (generated by the function
    !! ``date()`` from this module, default: ``.true.``).
    character(len=*), intent(in), optional :: title
    !! Title to print (default: empty).
    character(len=*), intent(in), optional :: fmt
    !! Format for printing (default: ``F20.12``).
    real(dp), intent(in), optional :: fac
    !! Scaling factor for printing.
    logical, intent(in), optional :: expand
    !! Expand the matrix in full form (default: ``.false.``)
    logical, intent(in), optional :: transpose
    !! Transpose the matrix for printing (default: ``.false.``)
    character(len=*), dimension(:), intent(in), optional :: clabels
    !! Labels for the columns.
    character(len=*), intent(in), optional :: file
    !! File in which the routine is called (default: empty).
    character(len=*), intent(in), optional :: proc
    !! Name of the procedure calling this routine (default: empty).

    character(len=MAX_STR_SIZE) :: to_print
    character(len=MAX_STR_SIZE) :: prtfmt
    integer :: nrows, ncols, bb, blocks, rem, ii
    logical :: texp, ttrans
    character(len=MAX_STR_SIZE) :: number_fmt
    integer :: i, j

    to_print = format_string(&
         & str=title, prefix=prefix, time_stamp=time_stamp, file=file, proc=proc&
         &)

    write(unit, '(A)') trim(to_print)

    nrows = size(data, 1)
    ncols = size(data, 2)

    if (present(fac)) then
       number_fmt = 'F20.12'
       if (present(fmt)) number_fmt = fmt
    else
       number_fmt = 'I20'
       if (present(fmt)) number_fmt = fmt
    end if

    texp = .false.
    if (present(expand)) texp = expand
    ttrans = .false.
    if (present(transpose)) ttrans = transpose

    ! Set the formatting
    if (texp) then
       if (ttrans) then
          write(prtfmt, '(A,I0,A,A)') '(', ncols, trim(number_fmt), ')'
       else
          write(prtfmt, '(A,I0,A,A)') '(', nrows, trim(number_fmt), ')'
       end if
    else
       write(prtfmt, '(A,A,A)') '(4', trim(number_fmt), ')'
    end if

    if (.not. texp) then
       if (ttrans) then
          nrows = size(data, 2)
          ncols = size(data, 1)
       else
          nrows = size(data, 1)
          ncols = size(data, 2)
       end if

       blocks = int(ncols / 4)
       rem = modulo(ncols, 4)

       ii = 1
       do bb=1, blocks
          if ((blocks > 1) .and. (rem > 0)) then
             if (present(clabels)) then
                write(unit, '(4A)') (adjustr(clabels(ii+i)), i=0, 3)
             else
                write(unit, '(4I20)') (ii+i, i=0, 3)
             end if
          else
             if (present(clabels)) then
                write(unit, '(4A)') (adjustr(clabels(ii+i)), i=0, 3)
             end if
          end if

          if (present(fac)) then
             do j=1, nrows
                if (ttrans) then
                   write(unit, fmt=prtfmt) (fac*data(ii+i, j), i=0, 3)
                else
                   write(unit, fmt=prtfmt) (fac*data(j, ii+i), i=0, 3)
                end if

             end do
             ii = ii + 4
          else
             do j=1, nrows
                if (ttrans) then
                   write(unit, fmt=prtfmt) (data(ii+i, j), i=0, 3)
                else
                   write(unit, fmt=prtfmt) (data(j, ii+i), i=0, 3)
                end if

             end do
             ii = ii + 4
          end if

       end do

       if (rem /= 0) then
          if ((blocks > 1) .and. (rem > 0)) then
             if (present(clabels)) then
                write(unit, '(4A)') (adjustr(clabels(ii+i)), i=0, 3)
             else
                write(unit, '(4I20)') (ii+i, i=0, 3)
             end if
          else
             if (present(clabels)) then
                write(unit, '(4A)') (adjustr(clabels(ii+i)), i=0, 3)
             end if
          end if

          if (present(fac)) then
             do j=1, nrows
                if (ttrans) then
                   write(unit, fmt=prtfmt) (data(ii+i, j), i=0, rem-1)
                else
                   write(unit, fmt=prtfmt) (data(j, ii+i), i=0, rem-1)
                end if

             end do
          else
             do j=1, nrows
                if (ttrans) then
                   write(unit, fmt=prtfmt) (data(ii+i, j), i=0, rem-1)
                else
                   write(unit, fmt=prtfmt) (data(j, ii+i), i=0, rem-1)
                end if

             end do
          end if

       end if

    else
       if (ttrans) then

          if (present(fac)) then
             do i=1, size(data, 2)
                write(unit, fmt=prtfmt) (fac*data(j, i), j=1, size(data, 1))
             end do
          else
             do i=1, size(data, 2)
                write(unit, fmt=prtfmt) (data(j, i), j=1, size(data, 1))
             end do
          end if

       else

          if (present(fac)) then
             do i=1, size(data, 1)
                write(unit, fmt=prtfmt) (fac*data(i, j), j=1, size(data, 2))
             end do
          else
             do i=1, size(data, 1)
                write(unit, fmt=prtfmt) (data(i, j), j=1, size(data, 2))
             end do
          end if

       end if
    end if
  end subroutine print_mat_i32_d2

end module mod_print_utils
