! Copyright (C) 2022  Light and Molecules Group

! This program is free software: you can redistribute it and/or modify
! it under the terms of the GNU General Public License as published by
! the Free Software Foundation, either version 3 of the License, or
! (at your option) any later version.

! This program is distributed in the hope that it will be useful,
! but WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
! GNU General Public License for more details.

! You should have received a copy of the GNU General Public License
! along with this program.  If not, see <https://www.gnu.org/licenses/>.

program nx_restart
  !! author: Baptiste Demoulin <baptiste.demoulin@univ-amu.fr>
  !! date: 2022-09-29
  !!
  !! Program to extract input files from Newton-X trajectory
  !!
  !! This program can be used to restart a trajectory, either by extending it, or by
  !! restarting it from any point for which an output is available.
  !!
  !! The functions used here are described in more details in the module
  !! ``[[mod_nx_extract]]``.
  use mod_kinds, only: dp
  use mod_nx_extract, only: main
  ! use mod_version, only: nx_version
  use iso_fortran_env, only: stdout => output_unit
  implicit none

  character(len=*), parameter :: THIS_NAME = 'nx_restart'

#include "version.f90"

  integer :: iarg, nr_args
  character(len=1024) :: arg
  logical :: skip_next

  integer  :: step, mode
  logical :: use_txt_files
  real(dp) :: extend_to
  character(len=:), allocatable :: h5file

  integer :: ierr

  print *, ''
  print *, 'Starting NEWTON-X RESTART v'//nx_version
  print *, ''

  step = -1
  extend_to = -1.0_dp
  
  ! Command line arguments
  nr_args = command_argument_count()
  
  use_txt_files = .false. ! HDF5 only !
  skip_next = .false.
  h5file = ''
  mode = 1
  do iarg=1, nr_args

     call get_command_argument(iarg, arg)
     if (len(trim(arg)) == 0) exit

     if (skip_next) then
        skip_next = .false.
        cycle

     else if ((arg == '--help') .or. (arg == '-h')) then
        call usage()
        stop

     else if ((arg == '--step') .or. (arg == '-s')) then
        call get_command_argument(iarg+1, arg)
        read(arg, *) step
        mode = 1
        skip_next = .true.

     else if (arg == '--txt') then
        use_txt_files = .true.

     else if (arg == '--h5file') then
        call get_command_argument(iarg+1, arg)
        h5file = trim(arg)
        skip_next = .true.

     else if ((arg == '--ext') .or. (arg == '-e')) then
        call get_command_argument(iarg+1, arg)
        read(arg, *) extend_to
        skip_next = .true.

      else if ((arg == '--mode') .or. (arg == '-m')) then
         call get_command_argument(iarg+1, arg)
         read(arg, *) mode
         skip_next = .true.
        
     else
        write(*, '(A)') 'Unrecognized option: '//trim(arg)
        write(*, '(A)') ''
        call usage()
        stop
     end if
  end do

  if (extend_to > 0 .and. step > 0) then
     print *, 'ERROR: --step and --ext are mutually exclusive !'
     call usage()
     stop
  end if

  print *, 'H5FILE: '//h5file

  ierr = main(step, extend_to, use_txt_files, h5file, mode)

  if (ierr == 0) then
     print *, ''
     print *, 'Normal termination of Newton-X extractor'
     print *, ''
  else
     print *, ''
     print *, 'ABNORMAL termination of Newton-X extractor'
     print *, ''
  end if

contains

  subroutine usage(output)
    integer, intent(in), optional :: output

    integer :: out

    character(len=*), parameter :: nl = NEW_LINE('c')

    out = stdout
    if (present(output)) out = output

    write(out, '(A)') 'Usage:'
    write(out, '(A)') '  '//THIS_NAME//' [-s step] [--txt] [--h5file H5FILE_NAME]'
    write(out, '(A)') ''
    write(out, '(A)') 'Options:'
    write(out, '(A)') '  -h, --help      Print this help.'
    write(out, '(A)') '  -s, --step      Step from which to extract the restart files.'
    write(out, '(A)') '                   By default, the restart files are created from'
    write(out, '(A)') '                   the last step in the trajectory.'
    write(out, '(A)') '  --txt,          Use the txt files for extracting the data.'
    write(out, '(A)') '                   This option is only useful when the code has '
    write(out, '(A)') '                   been compiled with the "USE_HDF5" options.'
    write(out, '(A)') '  --h5file,       Use the HDF5 file given as argument, instead'
    write(out, '(A)') '                   of the default one. This is useful to restart '
    write(out, '(A)') '                   from a backup file, when the default one is '
    write(out, '(A)') '                   corrupted.'
    write(out, '(A)') '  -e, --ext       Extends the simulation by the given value.'
    write(out, '(A)') '                   This value will be set as "tmax" in the input'
    write(out, '(A)') '                   file INFO_RESTART/nx-restart-config.nml.'
    write(out, '(A)') '                   CANNOT BE USED WITH --step.'
    write(out, '(A)') '  -m, --mode      Restart mode to be used.'
    write(out, '(A)') '                   1: Recomputes the last time step'
    write(out, '(A)') '                   2: Reds directly from the previous single point in TEMP'
    write(out, '(A)') '                   CANNOT BE USED WITH --step.'
    write(out, '(A)') '                   default: 1'
    write(out, '(A)') ''
    write(out, '(A)') '--------------- Examples ---------------'
    write(out, '(A)') ''
    write(out, '(A)') '  1) Extract files from the last simulation step:'
    write(out, '(A)') ''
    write(out, '(A)') '       '//THIS_NAME
    write(out, '(A)') ''
    write(out, '(A)') '  2) Extract files from step 100, using text files:'
    write(out, '(A)') ''
    write(out, '(A)') '       '//THIS_NAME//' -s 100 --txt'
    write(out, '(A)') ''
  end subroutine usage

end program nx_restart
